<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('throttle:60,1')->group(function () {
    Route::post('/activate', [App\Http\Controllers\Api\ActivationController::class, 'activate'])->middleware('throttle:5,1');
    Route::post('/failover/report', [App\Http\Controllers\Api\FailoverController::class, 'report']);
    Route::post('/payments/checkout', [App\Http\Controllers\Api\PaymentController::class, 'checkout']);
    Route::post('/payments/callback', [App\Http\Controllers\Api\PaymentController::class, 'callback']);

    Route::get('/live', function () {
        return App\Models\Channel::all();
    });

    Route::get('/notifications', [App\Http\Controllers\Api\NotificationController::class, 'index']);
    Route::put('/notifications/{id}/read', [App\Http\Controllers\Api\NotificationController::class, 'markAsRead']);
    Route::put('/notifications/read-all', [App\Http\Controllers\Api\NotificationController::class, 'markAllAsRead']);
    Route::delete('/notifications/{id}', [App\Http\Controllers\Api\NotificationController::class, 'destroy']);

    // ── IPTV Player Proxy Routes ──────────────────────────────────────────────
    // The iptv-player (Angular web app) needs these to bypass CORS when loading
    // external M3U playlists and Xtream / Stalker portal APIs.

    /**
     * Parse an M3U playlist from a remote URL and return it as-is.
     * GET /api/parse?url=http://...
     * Detects self-referential localhost URLs to avoid artisan serve deadlocks.
     */
    Route::get('/parse', function (Request $request) {
        $url = $request->query('url');
        if (!$url) {
            return response()->json(['error' => 'URL parameter is required'], 400);
        }

        // ── Self-referential guard ────────────────────────────────────────────
        // php artisan serve is single-threaded; calling ourselves would deadlock.
        // Detect if the URL points to our own server and serve it directly.
        $host = parse_url($url, PHP_URL_HOST) ?? '';
        $isSelf = in_array($host, ['localhost', '127.0.0.1']) &&
                  str_contains($url, '/api/');

        if ($isSelf) {
            // Strip query string, extract path after /api/
            $path = parse_url($url, PHP_URL_PATH);   // e.g. /api/user/playlist.m3u
            $query = parse_url($url, PHP_URL_QUERY); // e.g. token=xxx

            // Rebuild as an internal subrequest using our own controllers
            $subRequest = Request::create($path . ($query ? "?{$query}" : ''), 'GET');

            // Copy auth headers / token from query for playlist.m3u
            if ($query) {
                parse_str($query, $params);
                if (!empty($params['token'])) {
                    $subRequest->headers->set('Authorization', 'Bearer ' . $params['token']);
                }
            }

            // Route the internal request through Laravel's kernel
            $kernel = app(\Illuminate\Contracts\Http\Kernel::class);
            $response = $kernel->handle($subRequest);
            $content = $response->getContent();
        } else {
            // ── External URL ──────────────────────────────────────────────────────
            try {
                $response = \Illuminate\Support\Facades\Http::timeout(15)
                    ->withHeaders(['User-Agent' => 'Mozilla/5.0'])
                    ->get($url);
                $content = $response->body();
            } catch (\Exception $e) {
                return response()->json(['error' => $e->getMessage()], 500);
            }
        }

        // ── Parse M3U to JSON for iptv-player ─────────────────────────────────
        $lines = explode("\n", str_replace("\r", "", $content));
        $result = [
            'header' => ['attrs' => ['x-tvg-url' => ''], 'raw' => '#EXTM3U'],
            'items' => []
        ];

        $currentItem = null;
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (str_starts_with($line, '#EXTM3U')) {
                $result['header']['raw'] = $line;
            } elseif (str_starts_with($line, '#EXTINF:')) {
                $currentItem = [
                    'name' => '',
                    'tvg' => ['id' => '', 'name' => '', 'url' => '', 'logo' => '', 'rec' => ''],
                    'group' => ['title' => ''],
                    'http' => ['referrer' => '', 'user-agent' => ''],
                    'url' => '',
                    'raw' => $line
                ];

                if (preg_match('/tvg-id="([^"]*)"/', $line, $m)) $currentItem['tvg']['id'] = $m[1];
                if (preg_match('/tvg-name="([^"]*)"/', $line, $m)) $currentItem['tvg']['name'] = $m[1];
                if (preg_match('/tvg-logo="([^"]*)"/', $line, $m)) $currentItem['tvg']['logo'] = $m[1];
                if (preg_match('/group-title="([^"]*)"/', $line, $m)) $currentItem['group']['title'] = $m[1];

                $parts = explode(',', $line, 2);
                if (count($parts) > 1) {
                    $currentItem['name'] = trim($parts[1]);
                }
            } elseif (!str_starts_with($line, '#')) {
                if ($currentItem) {
                    $currentItem['url'] = $line;
                    $currentItem['raw'] .= "\n" . $line;
                    $result['items'][] = $currentItem;
                    $currentItem = null;
                }
            } elseif (str_starts_with($line, '#EXTVLCOPT:') || str_starts_with($line, '#EXTGRP:')) {
                if ($currentItem) {
                    $currentItem['raw'] .= "\n" . $line;
                    if (str_starts_with($line, '#EXTGRP:')) {
                        $currentItem['group']['title'] = trim(substr($line, 8));
                    }
                }
            }
        }

        return response()->json($result, 200, ['Access-Control-Allow-Origin' => '*']);
    });

    /**
     * Forward Xtream Codes API requests.
     * GET /api/xtream?url=...&username=...&password=...&action=...
     */
    Route::get('/xtream', function (Request $request) {
        $url    = $request->query('url');
        $params = $request->except('url');
        if (!$url) {
            return response()->json(['error' => 'URL is required'], 400);
        }

        try {
            $response = \Illuminate\Support\Facades\Http::timeout(20)
                ->withHeaders(['User-Agent' => 'Mozilla/5.0'])
                ->get($url, $params);

            return response()->json([
                'payload' => $response->json() ?? $response->body(),
            ], $response->successful() ? 200 : $response->status());
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    });

    /**
     * Forward Stalker Portal requests.
     * GET /api/stalker?url=...&macAddress=...
     */
    Route::get('/stalker', function (Request $request) {
        $url        = $request->query('url');
        $macAddress = $request->query('macAddress', '');
        $params     = $request->except(['url', 'macAddress']);
        if (!$url) {
            return response()->json(['error' => 'URL is required'], 400);
        }

        try {
            $response = \Illuminate\Support\Facades\Http::timeout(20)
                ->withHeaders([
                    'User-Agent' => 'Mozilla/5.0',
                    'Cookie'     => "mac={$macAddress}",
                ])
                ->get($url, $params);

            return response()->json([
                'payload' => $response->json() ?? $response->body(),
            ], $response->successful() ? 200 : $response->status());
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    });
});

Route::get('/movies', [App\Http\Controllers\Api\Admin\MovieController::class, 'index']);
Route::get('/series', [App\Http\Controllers\Api\Admin\SeriesController::class, 'index']);

// Admin Routes
Route::prefix('admin')->group(function () {
    Route::apiResource('channels', App\Http\Controllers\Api\Admin\ChannelController::class);
    Route::apiResource('resellers', App\Http\Controllers\Api\Admin\ResellerController::class);
    Route::apiResource('activation-codes', App\Http\Controllers\Api\Admin\ActivationCodeController::class);
    Route::apiResource('movies', App\Http\Controllers\Api\Admin\MovieController::class);
    Route::apiResource('series', App\Http\Controllers\Api\Admin\SeriesController::class);
    Route::get('/stats', function() {
        return response()->json([
            'active_users' => \App\Models\User::count(),
            'total_revenue' => \App\Models\Transaction::where('type', 'credit')->sum('amount'),
            'new_codes' => \App\Models\ActivationCode::where('status', 'inactive')->count(),
            'failover_events' => \App\Models\FailoverLog::count(),
        ]);
    });
});

// Reseller Routes
Route::prefix('reseller')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Api\Reseller\DashboardController::class, 'index']);
    Route::apiResource('codes', App\Http\Controllers\Api\Reseller\CodeController::class);
});

// User Auth Routes (Public)
Route::prefix('user')->group(function () {
    Route::post('/login', [App\Http\Controllers\Api\UserAuthController::class, 'login']);
    Route::post('/register', [App\Http\Controllers\Api\UserAuthController::class, 'register']);
});

// User Protected Routes (require Sanctum token)
Route::middleware('auth:sanctum')->prefix('user')->group(function () {
    Route::get('/profile', [App\Http\Controllers\Api\UserAuthController::class, 'profile']);
    Route::put('/profile', [App\Http\Controllers\Api\UserAuthController::class, 'updateProfile']);
    Route::put('/password', [App\Http\Controllers\Api\UserAuthController::class, 'changePassword']);
    Route::post('/logout', [App\Http\Controllers\Api\UserAuthController::class, 'logout']);
    Route::get('/subscriptions', [App\Http\Controllers\Api\UserAuthController::class, 'subscriptions']);

    // Notifications (user-scoped)
    Route::get('/notifications', [App\Http\Controllers\Api\NotificationController::class, 'userIndex']);
    Route::put('/notifications/read-all', [App\Http\Controllers\Api\NotificationController::class, 'markAllRead']);
    Route::put('/notifications/{id}/read', [App\Http\Controllers\Api\NotificationController::class, 'markAsRead']);
    Route::delete('/notifications/{id}', [App\Http\Controllers\Api\NotificationController::class, 'destroy']);

    // Preferences (onboarding + personalization)
    Route::get('/preferences', [App\Http\Controllers\Api\UserPreferencesController::class, 'show']);
    Route::put('/preferences', [App\Http\Controllers\Api\UserPreferencesController::class, 'update']);

    // Streaming (Watch button)
    Route::get('/stream-info', [App\Http\Controllers\Api\StreamController::class, 'streamInfo']);
});

// M3U Playlist - public route but validates token from query param
// (M3U players can't send Authorization headers)
Route::get('/user/playlist.m3u', function (\Illuminate\Http\Request $request) {
    // Accept token from query string for M3U player compatibility
    if ($request->has('token') && !$request->bearerToken()) {
        $request->headers->set('Authorization', 'Bearer ' . $request->query('token'));
    }

    // Authenticate manually
    $user = auth('sanctum')->user();
    if (!$user) {
        return response('# Unauthorized', 401, ['Content-Type' => 'audio/mpegurl']);
    }

    $request->setUserResolver(fn() => $user);
    return app(\App\Http\Controllers\Api\StreamController::class)->playlist($request);
});

