<?php

namespace App\Services;

use App\Models\Notification;

class NotificationService
{
    /**
     * Send a notification to the admin panel.
     */
    public static function admin(string $title, string $message, string $type = 'info', string $icon = '🔔', array $meta = []): Notification
    {
        return Notification::create([
            'user_id' => null,
            'title'   => $title,
            'message' => $message,
            'type'    => $type,
            'target'  => 'admin',
            'icon'    => $icon,
            'meta'    => $meta,
            'is_read' => false,
        ]);
    }

    /**
     * Send a notification to a specific user.
     */
    public static function user(int $userId, string $title, string $message, string $type = 'info', string $icon = '🔔', array $meta = []): Notification
    {
        return Notification::create([
            'user_id' => $userId,
            'title'   => $title,
            'message' => $message,
            'type'    => $type,
            'target'  => 'user',
            'icon'    => $icon,
            'meta'    => $meta,
            'is_read' => false,
        ]);
    }

    /**
     * Send a notification to BOTH admin and user (creates 2 records).
     */
    public static function both(int $userId, string $title, string $message, string $type = 'info', string $icon = '🔔', array $meta = []): void
    {
        static::admin($title, $message, $type, $icon, $meta);
        static::user($userId, $title, $message, $type, $icon, $meta);
    }

    /**
     * Broadcast to all users who have a matching preference.
     * e.g. notify all users who favorited "Liverpool" when a Liverpool match starts.
     */
    public static function broadcastByPreference(
        string $preferenceKey,
        string $preferenceValue,
        string $title,
        string $message,
        string $type = 'info',
        string $icon = '🔔',
        array $meta = []
    ): int {
        $users = \App\Models\UserPreference::where($preferenceKey, $preferenceValue)
            ->with('user')
            ->get();

        $count = 0;
        foreach ($users as $pref) {
            if ($pref->user) {
                static::user($pref->user->id, $title, $message, $type, $icon, $meta);
                $count++;
            }
        }

        return $count;
    }
}
